/****************************************************************
*	                                                            *
*  Steuerprogramm fr 2m Funkmodul DRA818V	LED Anzeigeversion	*
* 									    						*
*  Meinrad Gtz                                                 *
*                                                               *
*  Version 1.0                 8. September 2017                *
*                                                               *
*  Programmbeschreibung:                                        *
*                                                               *
*									    						*
*  Ansteuerung eines Funkmoduls DRA818V (2m) ber die Serielle	*
*  Schnittstelle. 							    				*
* (9600 Baud 1 Start Bit, 8 Data Bits, 1 Stop Bit, No Parity)	*
*									    						*
*  Es werden initial eingestellt:					    		*
*	Sendefrequenz = Empfangsfrequenz: 145,500 MHz			    *
*	Keine Ablage							    				*
*	Squelch = 1							    					*
*	Volume = 5							    					*
*									    						*
*	ber Tasten kann eingestellt werden:				    	*
*	Taste Up, Tatse Down:										*
					+- 25kHz der Empfangsfrequenz im Bereich:   *
*					145,500MHz - 145,800MHz			    		*
*	Taste 12,5kHZ:	Erhhung der Empfangsfrequenz um 12,5kHz    *
*					gegenber der mit den Tasten Up, Down	    *
*					eingestellten Frequenz			    		*
*	Taste Ablage:	Sendefrequenz um 600kHz niedriger als die   *
*					Empfangsfrequenz (Relaisbetrieb)	    	*
*	Taste 1750Hz:	Ausgabe eines 1750 Hz (Relais-) Ton	    	*
*									    						*
*									    						*
*  Der Betriebszustand wird durch 6 LEDs wie folgt angezeigt:	*
*	4 LEDs zur Frequenz (Kanal) Anzeige im Dualcode:		    *
*	0 0 0 0 (0) = 145,500MHz					    			*
*	0 0 0 1 (1) = 145,525MHz					    			*
*									    						*
*	1 0 1 1 (11) = 145,775MHz					    			*
*	1 1 0 0 (12) = 145,800MHz					    			*
*									    						*
*  Wenn die Frequenz um +12,5kHz angehoben wird, so wird dies	*
*  durch eine LED angezeigt.							    	*
*  Eine aktive Ablage (-600 kHz) wird durch eine LED angezeigt.	*
*									    						*
*  Prozessor:		ATMEGA 8L						    		*
*  Taktfrequenz:	3,686 MHz					    			*
*                                                               *
****************************************************************/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>

//Defines
#define TIMER0_10MS		144		//Ladewert Timer 0 fr 10ms
#define TIMER0_1750		127 	//Ladewert Timer 0 fr 1750 Hz

#define ENTPRELLZEIT  4			//Tasten Entprellzeit = TIMER0_10MS * ENTPRELLZEIT
#define TASTENMASKE	  0b01111100

#define TON_1750	PORTB2	//1750 Hz Signalausgang

#define TRUE		1
#define FALSE		0

#define TON			0
#define TIMER		1

#define MAXKANAL	12	//Maximal 12 Kanle (0 -> 145,500MHz; 1 -> 145,525MHz; ... 11 -> 145,775MHz; 12 -> 145,800MHz)


//Function declaration
void Init_Timer0(void);
void Convert_US_Int_Bcd(unsigned long int Zahl);
void Convert_BCD_ASCII(void);
void Init_CPU(void);
void Pause_Timer (unsigned char Zeit);

void Init_UART(void);
void Send_CR_LF (void);
void Send_String (unsigned char *pSource, unsigned char cAnzahl);
void Receive_String (unsigned char *pDestination, unsigned char cAnzahl);

void Init_DRA818 (void);
unsigned char Send_SetGroup(unsigned char cKanalNr, unsigned char cAblage, unsigned char cStepp_12_5);
unsigned char Send_VolumeCommand(unsigned char cVol);
unsigned char Send_FilterCommand(unsigned char cDeemph, unsigned char cHighpass, unsigned char cLowpass);

void Set_LED(unsigned char cKanalLED, unsigned char cAblageLED,unsigned char cSteppLED);

//Globales
unsigned char cSekunde;					//Kennzeichnungsflag fr Sekunde
unsigned char cSecTakt;					//Sekundenzhler;
unsigned char cPausenTime;				//Zhlvariable fr Pausenzeiten
unsigned char cPausenTakt;				//Arbeitsvariable fr PausenTime (sec. Takt)
unsigned char cTimerBetriebsart;		//Timer Betriebsart: 10ms Timer / 1750 Hz Timer

unsigned char BCD_Numb[11];				//BCD Ergebnisspeicher
unsigned long int uliEmpfangsfrequenz = 145500000;	//Empfangsfrequenz
unsigned long int uliSendefrequenz = 145500000;		//Sendefrequenz
unsigned long int uliAblage = 600000;				//Frequenzablage
unsigned char cKanalNr = 0;
unsigned char cStepp_12_5 = 0;
unsigned char cAblage = 0;				//0 -> 0 kHz Ablage; 1 -> 600 kHz Ablage
unsigned char cBandbreite = 0;			//0 -> 12,5 kHz; 1 -> 25 kHz
unsigned char cSpuelch = 1;				//0 = off (1 - 8)
unsigned char cVolumen = 5;

unsigned char cPrellcounter;
unsigned char cTasten_Neu;
unsigned char cTasten_Alt;
unsigned char cTastenzustand;
unsigned char cTaste_Up;
unsigned char cTaste_Down;
unsigned char cTaste_1750;
unsigned char cTaste_12_5;
unsigned char cTaste_Ablage;

unsigned char Receive_Buffer[20];

		
int main(void)
{
	unsigned char cHelp;

	Init_CPU();		//Initialisierung
	Init_Timer0();
	Init_UART();

	wdt_enable(WDTO_2S);

	Set_LED(15,1,1);	//alle LEDs ON	
	Pause_Timer(2);		//Startphase 2 sec
	Set_LED(0,0,0);		//alle LEDs OFF
	Pause_Timer(1);
	Set_LED(15,1,1);	//alle LEDs ON	
	Pause_Timer(1);
	Set_LED(0,0,0);		//alle LEDs OFF
	
	Init_DRA818();

	cKanalNr = 0;		//Empfangsfrequenz = Sendefrequenz --> 145,500MHz
	cAblage = 0;
	cStepp_12_5 = 0;

	Set_LED(cKanalNr, cAblage,cStepp_12_5);

	cHelp = Send_SetGroup(cKanalNr, cAblage, cStepp_12_5);
	if (TRUE == cHelp)
	{
		cHelp = Send_VolumeCommand(5);			//Volume 5
		if (TRUE == cHelp)
		{
			cHelp = Send_FilterCommand(0,0,0);		//PRE/DE-EMPH = On; Highpass = On; Lowpass = On; (1 schaltet die Funktion aus)
		}
	}
	if (FALSE == cHelp)
	{
		while (1)
		{
			Set_LED(15,1,1);		//alle LEDs ON -> Fehlermeldung (Neustart notwendig)
			Pause_Timer(1);
			Set_LED(0,0,0);			//alle LEDs OFF
			Pause_Timer(1);
		}
	}

	while (1)
	{
		if (cTaste_Up == 1)
		{
			cKanalNr++;
			if (cKanalNr > MAXKANAL)
			{
				cKanalNr = MAXKANAL;
			}
			Set_LED(cKanalNr, cAblage,cStepp_12_5);
			cHelp = Send_SetGroup(cKanalNr, cAblage, cStepp_12_5);	//Gruppeneinstellung an Modul bertragen
			while (cTaste_Up == 1)
			{
				cHelp++;				//Warten bis Taste Up losgelassen (entprellt) ist
			}
		}

		if (cTaste_Down == 1)
		{
			cKanalNr--;
			if (cKanalNr == 255)
			{
				cKanalNr = 0;
			}
			Set_LED(cKanalNr, cAblage, cStepp_12_5);
			cHelp = Send_SetGroup(cKanalNr, cAblage, cStepp_12_5);	//Gruppeneinstellung an Modul bertragen
			while (cTaste_Down == 1)
			{
				cHelp++;				//Warten bis Taste Down losgelassen (entprellt) ist
			}
		}

		if (cTaste_12_5 == 1)
		{
			if (cStepp_12_5 == 1)
				cStepp_12_5 = 0;
			else
				cStepp_12_5 = 1;

			Set_LED(cKanalNr, cAblage,cStepp_12_5);
			cHelp = Send_SetGroup(cKanalNr, cAblage, cStepp_12_5);	//Gruppeneinstellung an Modul bertragen
			while (cTaste_12_5 == 1)
			{
				cHelp++;				//Warten bis 12,5 kHz Taste losgelassen (entprellt) ist
			}
		}

		if (cTaste_Ablage == 1)
		{
			if (cAblage == 1)
				cAblage = 0;
			else
				cAblage = 1;

			Set_LED(cKanalNr, cAblage,cStepp_12_5);
			cHelp = Send_SetGroup(cKanalNr, cAblage, cStepp_12_5);	//Gruppeneinstellung an Modul bertragen
			while (cTaste_Ablage == 1)
			{
				cHelp++;				//Warten bis Ablage Taste losgelassen (entprellt) ist
			}
		}


		if (cTaste_1750 == 1)
		{
			//1750 Hz Ton solange wie CE-Taste gedrckt ist
			TCCR0 = 0b00000000;			//Timer 0 Stop, damit Timer fr Tonerzeugung genutzt werden kann
			TCNT0 = 256 - TIMER0_1750;	//Ladewert fr 1750 Hz Ton
			cTimerBetriebsart = TON;
			TCCR0 = 0b00000010;					//Timer 0 Starten (Takt = CPU-Takt)
			cHelp = PIND;				//1750 Hz-Taste abfragen
					cHelp = cHelp & 0b00010000;
					while (cHelp == 0)
					{
						cHelp = PIND;							//1750Hz-Taste abfragen
						cHelp = cHelp & 0b00010000;
						wdt_reset();
					}
					TCCR0 = 0b00000000;				//Timer 0 Stop, und auf 10ms Timer zurckschalten
					TCNT0 = 256 - TIMER0_10MS;		//Ladewert fr 10ms
					cTimerBetriebsart = TIMER;
					TCCR0 = 0b00000100;				//Timer 0 wieder Starten (Takt = CPU-Takt/256)
					cSecTakt = 0;					//Sekundentakt = 0
					cSekunde = 0;
					PORTB = PORTB & ~(1 << TON_1750);	//Port 1750 Hz Ton auf 0 (Einschwingverhalten)
					while (cTaste_1750 == 1)
					{
						cHelp++;				//Warten bis 1750 Hz Taste losgelassen (entprellt) ist
					}			
		}

		wdt_reset();
	}
}	


/****************************************************************
*																*
*	Initialisierung CPU 										*
*																*
*****************************************************************/
void Init_CPU(void)
{
	DDRB = 0x07;		//PB7:In   PB6:In   PB5:In   PB4:In   PB3:In  PB2:Out  PB1:Out  PB0:Out
	PORTB = 0x38;		//PB7:k.PU PB6:k.PU PB5:PU   PB4:PU   PB3:PU  PB2:0    PB1:0    PB0:0

						//Alle LEDs ON
	DDRC = 0x3F;		//x   PC6:IN   PC5:Out  PC4:Out  PC3:Out  PC2:Out  PC1:Out  PC0:Out
	PORTC = 0x00;		//x   PC6:k.PU PC5:0    PC4:0    PC3:0    PC2:0    PC1:0    PC0:0 

	DDRD = 0xFF;		//PD7:IN PD6:IN PD5:IN PD4:IN PD3:IN PD2:IN PD1:Out PD0:IN
	PORTD = 0xFF; 		//PD7:PU PD6:PU PD5:PU PD4:PU PD3:PU PD2:PU PD1:1   PD0:PU

	cPrellcounter = 0;
	cTasten_Neu = PIND;				//Tastatur initialisieren
	cTasten_Neu = cTasten_Neu & TASTENMASKE;
	cTasten_Alt = cTasten_Neu;
	cTastenzustand = TASTENMASKE;	//keine Taste gedrckt
	cTaste_Up = 0;
	cTaste_Down = 0;
	cTaste_1750 = 0;
	cTaste_12_5 = 0;
	cTaste_Ablage = 0;

	wdt_enable(WDTO_2S);
	sei();						//Interrupts freigeben
}


void Init_DRA818 (void)
{
	unsigned char Init_String[] = "AT+DMOCONNECT";
	
	Send_String (&Init_String[0], 13);
	Send_CR_LF ();
	Receive_String (&Receive_Buffer[0], 15);
}



/****************************************************************
*	Unsigned Integer auf BCD  Convertierung						*
*																*
*	Der bergebene unsigned long Integerwert wird in eine       *
*	10-stellige BCD Zahl gewandelt.								*
*	Das Ergebnis steht in BCD_Numb[] 							*
****************************************************************/
void Convert_US_Int_Bcd(unsigned long int Zahl)           //10-stellig
{
	BCD_Numb[10] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[9] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[8] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[7] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[6] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[5] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[4] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[3] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[2] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[1] = Zahl % 10;
}      




/****************************************************************
*	BCD auf ASCII Convertierung									*
*																*
*	Die in BCD_Numb[] stehende 10-stellige BCD Zahl wird auf 	*
*	ASCII gewandelt und wieder in BCD_Numb[]gespeichert.		*
*   Das auf BCD_Numb[0] stehende Vorzeichen ist schon auf ASCII.*
****************************************************************/
void Convert_BCD_ASCII(void)
{
	unsigned char i;

	for (i = 1; i < 11; i++)
	{
		BCD_Numb[i] = BCD_Numb[i] + 0x30;
	}
}


/*	Initialisierung Timer 0	auf 10ms	*/
void Init_Timer0(void)
{
	cTimerBetriebsart = TIMER;		//Timer 0 als 10 ms Timer
	TCCR0 = 0b00000000;			//Timer 0 Stop
	TIMSK = (1 << TOIE0);		//Timer 0 Interrupt freigeben
	TCNT0 = 256 - TIMER0_10MS;	//Ladewert fr 10ms
	TCCR0 = 0b00000100;			//Timer 0 Starten (Takt = CPU-Takt/256)
	cSecTakt = 0;				//Sekundentakt = 0
	cSekunde = 0;
}



/* Funktion zur Erzeugung einer Wartezeit im Sekundentakt
   maximale Wartezeit: 255 sec. Steuerung ber Timer 0
   Wenn inerhalb der Wartezeit eine beliebige Taste gedrckt wird
   wird die Wartezeit abgebrochen. */
void Pause_Timer (unsigned char Zeit)
{
	unsigned char Tastenzustand = TASTENMASKE;

	cPausenTakt = 0;			//Pausentaktsteuerung synchronisieren
	cPausenTime = Zeit;			//Zeit bernehmen
	while (cPausenTime != 0 && Tastenzustand == TASTENMASKE)
	{
		//Warten bis Zeit abgelaufen ist
		Tastenzustand = PIND;
		Tastenzustand = Tastenzustand & TASTENMASKE;
		wdt_reset();
	}
}


/* Interrupt Service Routinen	*/

ISR(TIMER0_OVF_vect)
{
	if (TON == cTimerBetriebsart)
	{
		TCNT0 = 256 - TIMER0_1750;	//Ladewert fr 1750 Hz
		PORTB^=(1<<TON_1750);
	}
	else
	{
		TCNT0 = 256 - TIMER0_10MS;	//Ladewert fr 10ms

		wdt_reset();

		cSecTakt++;
		if (cSecTakt == 60)
		{
			cSekunde = 1;			//Sekundentakt
			cSecTakt = 0;
		}

		cPausenTakt++;
		if (cPausenTakt == 60)
		{
			cPausenTakt = 0;			//Pausentakt Steuerung
			cPausenTime--;
		}

		cTasten_Neu = PIND;			//Tastatur einlesen
		cTasten_Neu = cTasten_Neu & TASTENMASKE;
		if (cTasten_Neu == cTasten_Alt)
		{
			cPrellcounter++;
			if(cPrellcounter == ENTPRELLZEIT)
			{
				cPrellcounter = 0;
				cTastenzustand = cTasten_Neu;
				if((cTastenzustand & 0b00000100) == 0)
				{
					cTaste_Up = 1;
				}
				else
				{
					cTaste_Up = 0;
				}
				if((cTastenzustand & 0b00001000) == 0)
				{
					cTaste_Down = 1;
				}
				else
				{
					cTaste_Down = 0;
				}
				if((cTastenzustand & 0b00010000) == 0)
				{
					cTaste_1750 = 1;
				}
				else
				{
					cTaste_1750 = 0;
				}
				if((cTastenzustand & 0b00100000) == 0)
				{
					cTaste_12_5 = 1;
				}
				else
				{
					cTaste_12_5 = 0;
				}
				if((cTastenzustand & 0b01000000) == 0)
				{
					cTaste_Ablage = 1;
				}
				else
				{
					cTaste_Ablage = 0;
				}
			}
		}
		else
		{
			cTasten_Alt = cTasten_Neu;
			cPrellcounter = 0;
		}
	}
}


/********************************************************************
*																	*
*	LEC Anzeigefunktion												*
*																	*
********************************************************************/
void Set_LED(unsigned char cKanalLED, unsigned char cAblageLED,unsigned char cSteppLED)
{
	unsigned cHelp;

	cKanalLED =~cKanalLED;

	cHelp = cKanalLED & 0b00001111;

	if (cAblageLED == 0)
	{
		cHelp = cHelp | 0b00100000;	
	}

	if (cSteppLED == 0)
	{
		cHelp = cHelp | 0b00010000;
	}

	PORTC = cHelp;
}



/********************************************************************
*																	*
*	UART Funktionen													*
*																	*
********************************************************************/
void Init_UART(void)
{
	/* Set baud rate to 9600 Baud */
	UBRRH = 0;
	UBRRL = 23;
	/*  11 -> 19,2 kBaud
		23 -> 9600 Baud
		47 -> 4800 Baud 
   		95 -> 2400 Baud */

	/* Enable receiver and transmitter */
	UCSRB = (1<<RXEN)|(1<<TXEN);

	/* Enable only transmitter */
	//UCSRB = (1<<TXEN);

	/* Set frame format: 8data, 1stop bit */
	UCSRC = (1<<URSEL)|(3<<UCSZ0);
}


void Send_UART (unsigned char cData)
{
	// Wait if a byte is being transmitted
	while((UCSRA&(1<<UDRE)) == 0);

	// Transmit data
	UDR = cData;
	wdt_reset();
}

void Send_CR_LF (void)
{
	Send_UART (0x0D);		//CR
	Send_UART (0x0A);		//LF
}


void Send_String (unsigned char *pSource, unsigned char cAnzahl)
{
	unsigned char i;

	while (cAnzahl != 0)
	{
		i = *pSource;
		Send_UART(i);
		pSource++;
		cAnzahl--;
	}
}

unsigned char Receive_UART (void)
{
	/* Wait for data to be received */
	while ( !(UCSRA & (1<<RXC)) );

	wdt_reset();
	/* Read data from buffer */
	return UDR;
}

void Receive_String (unsigned char *pDestination, unsigned char cAnzahl)
{
	unsigned char i;

	while (cAnzahl != 0)
	{
		i = Receive_UART ();
		*pDestination = i;
		pDestination++;
		cAnzahl--;
	}
}

unsigned char Send_SetGroup(unsigned char cKanalNr, unsigned char cAblage, unsigned char cStepp_12_5)
{
	unsigned long int uliRx_Frequenz, uliTx_Frequenz;
	unsigned long int uliKanalNr, uliAblage, uliStepp;

	uliKanalNr = cKanalNr;
	uliAblage = cAblage;
	uliStepp = cStepp_12_5;

	//AT+DMOSETGROUP=GBW,TFV, RFV,Tx_CTCSS,SQ,Rx_CTCSS<CR><LF>
	unsigned char Init_String[] = "AT+DMOSETGROUP=";
	Send_String (&Init_String[0], 15); //Sendebefehl Group
	Send_UART ('0'); //Bandbreite (0 -> 12,5 kHz; 1 -> 25kHz)
	Send_UART (',');
 
	uliTx_Frequenz = (145500000 + uliKanalNr * 25000 + uliStepp * 12500) - uliAblage * 600000;

	Convert_US_Int_Bcd(uliTx_Frequenz); //Sendefrequenz
	Convert_BCD_ASCII();
	Send_UART (BCD_Numb[2]);         
	Send_UART (BCD_Numb[3]);         
	Send_UART (BCD_Numb[4]);         
	Send_UART ('.'); 
	Send_UART (BCD_Numb[5]); 
	Send_UART (BCD_Numb[6]); 
	Send_UART (BCD_Numb[7]); 
	Send_UART (BCD_Numb[8]);
	Send_UART (',');
 
	uliRx_Frequenz = 145500000 + uliKanalNr * 25000 + uliStepp * 12500;
 
	Convert_US_Int_Bcd(uliRx_Frequenz); //Empfangsfrequenz
	Convert_BCD_ASCII();
	Send_UART (BCD_Numb[2]); 
	Send_UART (BCD_Numb[3]); 
	Send_UART (BCD_Numb[4]); 
	Send_UART ('.'); 
	Send_UART (BCD_Numb[5]); 
	Send_UART (BCD_Numb[6]); 
	Send_UART (BCD_Numb[7]); 
	Send_UART (BCD_Numb[8]);
	Send_UART (',');
	Send_UART ('0'); //Tx_CTCSS abgeschaltet
	Send_UART ('0');
	Send_UART ('0');
	Send_UART ('0');
	Send_UART (',');
	Send_UART ('1'); //Squelch (0 = Off; 1 - 9)
	Send_UART (',');
	Send_UART ('0'); //Rx_CTCSS abgeschaltet
	Send_UART ('0');
	Send_UART ('0');
	Send_UART ('0');
	Send_CR_LF ();
	Receive_String (&Receive_Buffer[0], 16);
	if ('0' == Receive_Buffer[13])
		return TRUE;
	else
		return FALSE;
}


unsigned char Send_VolumeCommand(unsigned char cVol)
{
	//AT+DMOSETVOLUME=x <CR><LF>
	unsigned char Volume_String[] = "AT+DMOSETVOLUME=";
	
	Send_String (&Volume_String[0], 16);		//Sendebefehl Volume
	Send_UART (cVol + 0x30);					//Volume als ASCII
	Send_CR_LF ();

	Receive_String (&Receive_Buffer[0], 17);
	Receive_Buffer[15] = ' ';	//CR LF aus Empfangspuffer
	Receive_Buffer[16] = ' ';	//(wegen Anzeige) lschen.

	if ('0' == Receive_Buffer[14])
		return TRUE;
	else
		return FALSE;

}
	

unsigned char Send_FilterCommand(unsigned char cDeemph, unsigned char cHighpass, unsigned char cLowpass)
{
	//AT+SETFILTER=PRE/DE-EMPH,Highpass,Lowpass <CR><LF>
	unsigned char Filter_String[] = "AT+SETFILTER=";
	
	Send_String (&Filter_String[0], 13);		//Sendebefehl Filter
	Send_UART (cDeemph + 0x30);					//Deemphasis als ASCII
	Send_UART (',');
	Send_UART (cHighpass + 0x30);				//Highpass als ASCII
	Send_UART (',');
	Send_UART (cLowpass + 0x30);				//Lowpass als ASCII
	Send_CR_LF ();
	
	Receive_String (&Receive_Buffer[0], 17);
	Receive_Buffer[15] = ' ';	//CR LF aus Empfangspuffer
	Receive_Buffer[16] = ' ';	//(wegen Anzeige) lschen.

	if ('0' == Receive_Buffer[14])
		return TRUE;
	else
		return FALSE;
}
